//==============================================================================
// Project:		Wealth Transfers and their Economic Effects
// File name:	Inheritance confidence intervals
// Objective: 	Generate confidence intervals for the aggregate value of
//				inheritances received each year.
//
// Created: 	24/09/2021
// Modified: 	15/11/2021
//==============================================================================

*************************************
*** Loading and cleaning the data ***
*************************************

clear	// clearing the workspace

cd "XXXX"	// set the content directory here

use HILDA_restricted_combined_rescaled	// importing the unconfidentialised HILDA data for waves 1-19 that includes the new weights for residents of non-private dwellings

drop if hgint == 0	// dropping non-respondings persons

by wave, sort: egen rp_sample_size = count(benchmarked_new_hhwtrp) if benchmarked_new_hhwtrp > 0				// calculating the responding-person sample size
by wave, sort: egen rp_reference_pop = sum(benchmarked_new_hhwtrp)	// calculating the responding-person reference population

gen inheritance = oifinha							// generating an inheritance variable
replace inheritance = 0 if inheritance <= 0			// replacing negative inheritance values as zero

*********************
*** Data analysis ***
*********************

svyset [pweight = benchmarked_new_hhwtrp], strata(xhhstrat) psu(xhhraid)	// setting the weights, strata and primary sampling unit

svy: mean inheritance, over(wave)					// calcuating the mean inheritance by wave

matrix results = r(table)'							// saving the results to a matrix called 'results'

keep wave rp_sample_size rp_reference_pop			// keeping key variables

collapse rp_sample_size rp_reference_pop, by(wave)	// collapsing the dataset

save rp_reference_pop.dta, replace					// saving the dataset

clear	// clearing the workspace

svmat results	// creating a dataset from the saved results

rename results1 mean			// mean inheritance
rename results3 test_statistic	// test statistic
rename results4 p_value			// p-value
rename results5 lower_CI		// lower bound of 95% confidence interval
rename results6 upper_CI		// upper bound of 95% confidence interval
rename results7 df				// degrees of freedom
rename results8 critical_value	// critical value

drop results9	// dropping 'results9', which is a vector of zeroes

// generating a wave variable
gen wave = .
forvalues i = 1/19 {
    replace wave = `i' in `i'
}

save confidence_intervals.dta, replace				// saving the dataset

merge 1:1 wave using rp_reference_pop.dta, nogen	// merging in the reference population figures

merge 1:1 wave using total_inheritances.dta, nogen	// merging in the total inheritances figures

// generating a year variable
rename wave year
replace year = year + 2000

local vars mean lower_CI upper_CI	// creating a local macro including the mean and confidence intervals

// scaling the individual estimates up to the population
foreach j in `vars' {
    gen pop_`j' = `j' * rp_reference_pop
}

merge 1:1 year using CPI_FY_2018.dta, nogen	// merging in the consumer price index figures with 2018 as the base year

drop if (year == 2001 | year == 2020)		// dropping 2020 (no observations) and 2001 (very few observations)

save confidence_intervals.dta, replace		// saving the dataset

local pop_vars pop_mean pop_lower_CI pop_upper_CI total_inheritances	// creating a local macro including the mean, sum and confidence intervals

// converting the population estiamtes to constant 2018 prices
foreach k in `pop_vars' {
    gen `k'_2018 = `k' * (100 / cpi)
}

*************************************
*** Saving and exporting the data ***
*************************************

keep year pop_lower_CI_2018 total_inheritances_2018 pop_upper_CI_2018	// keeping the key variables

order year pop_lower_CI_2018 total_inheritances_2018 pop_upper_CI_2018	// re-ordering the variables

save confidence_intervals_2018.dta, replace	// saving the dataset